#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>

#include <sys/param.h>
#include <sys/mman.h>

#define DEBUG_ON

#include "libpdpu.h"

/*
 * This function configures and sets up the AXI DMA Channel.
 * @channel: The DMA channel structure (either RX or TX).
 * return status OK if the initialization is complete else status ERR
 */
int AXI_DMA_Init(Channel *channel) {
    char chan_name[64] = "/dev/";
    strcat(chan_name, channel->name);

#ifdef DEBUG_ON
    printf("[INFO] Starting DMA channel: %s\n", chan_name);
#endif

    // Open the channel
    channel->fd = open(chan_name, O_RDWR);
    if (channel->fd == -1) {
#ifdef DEBUG_ON
        perror("[ERR] Cannot open DMA channel");
#endif
        return ERR;
    }

    // Map memory
    /* Note: The size here should be adjusted based on actual needs. */
#ifdef DEBUG_ON
    printf("[INFO] Allocating memory for the DMA buffers\n");
#endif

    channel->buf_ptr = (struct channel_buffer *)mmap(NULL, sizeof(struct channel_buffer), PROT_READ | PROT_WRITE, MAP_SHARED, channel->fd, 0);
    if (channel->buf_ptr == MAP_FAILED) {
#ifdef DEBUG_ON
        perror("[ERR] Cannot allocate memory for the DMA buffers");
#endif
        close(channel->fd);
        return ERR;
    }

#ifdef DEBUG_ON
    printf("[INFO] DMA channels initialized successfully\n");
#endif

    return OK;
}

/* end of AXI_DMA_Init() */

/*
 * This function writes data to a memory-mapped AXI Lite register.
 * Basically, this function writes 32-bit wide data to virtual memory at address = reg + offset.
 * @reg: The base address of the AXI Lite register.
 * @offset: The offset of the AXI Lite register relative to the base.
 * @data: The unsigned 32-bit long data to write.
 */
void AXILite_Register_Write(uint32_t *reg, uint32_t offset, uint32_t data) {
    reg[offset >> 2] = data;
} /* end of AXILite_Register_Write() */

/*
 * This function reads 32-bit wide data from a memory-mapped AXI Lite register at address = reg + offset.
 * @reg: The base address of the AXI Lite register.
 * @offset: The offset of the AXI Lite register relative to the base address.
 * return The value stored in the register.
 */
uint32_t AXILite_Register_Read(uint32_t *reg, uint32_t offset) {
    return reg[offset >> 2];
} /* end of AXILite_Register_Read() */

/*
 * This function is used to call the fpgautil tools for dynamic FPGA programming
 * from user-space. You can program only the bitstream, or you can also load the device tree overlay.
 * @firmware_bin: The full path to the bitstream in .bin format (e.g. /lib/firmware/<firmware>/<firmware>.bin).
 * @firmware_dtbo: The full path to the device tree overlay compiled object file (e.g. /lib/firmware/<firmware>/<firmware>.dtbo).
 * Note: In case only the bitstream is to be used set the firmware_dtbo to NULL
 */
void program_fpga(char *firmware_bin, char *firmware_dtbo)
{
  char command[2048];

  if (firmware_dtbo == NULL) {
    snprintf(command, 2048, "sudo fpgautil -b %s -f Full", firmware_bin);
  } else {
    snprintf(command, 2048, "sudo fpgautil -b %s -o %s -f Full", firmware_bin, firmware_dtbo);
  }

  system(command);
}
