/*
 * Creator     	: 	Ronaldo Tsela
 * Description 	: 	This program is the software interface to the Dummy hardware component.
 * 					This peripheral is designed to perform DMA transactions between the PS and PL
 * 					by writing to and reading from a controllable number of words using AXI4 Lite registers.
 * Compilation  : 	arm-linux-gnueabihf-gcc -g -Wall -o dummy-worker dummy-main.c dummy.c libpdpu.c -lpthread
 * Run 			:   ./dummy-worker <file in> <file out> <words in> <words out>
 * Input		:	file in  : Input file name/directory.
 * 					file out : Output file name/directory.
 * 					words in : Number of words to transfer to the PL.
 * 					words out: Number of words to receive from the PL.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <pthread.h>
#include <stdint.h>
#include <string.h>
#include <fcntl.h>
#include <sys/param.h>
#include <sys/mman.h>

#include "dummy.h"
#include "pdpu_worker.h"

#define DEBUG_ON

int main (int argc, char *argv[]) {

	char *fin_name, *fout_name;
	int fdi, fdo, fd_reg;

	int count_in, count_out;

	int halt_op = 0;

/*----------Step 1: Get, Check and Verify the User Input Parameters -------------------------------*/
	
	if (argc < 5 || strcmp(argv[1], "h") == 0) {
#ifdef DEBUG_ON
		printf("Usage  : %s <file in> <file out> <words in> <words out>\n\n", argv[0]);
		printf("\t<file in>   : Input file name/directory from where to read the data to transfer to the PL side\n");
		printf("\t<file out>  : Output file name/directory where to write the data from the PL side\n");
		printf("\t<words in>  : Number of words to transfer to the PL\n");
		printf("\t<words out> : Number of words to receive from the PL\n");
		printf("\nHelp   : %s h\n\n", argv[0]);
#else
		printf("%d\n", STATUS_ERROR);
#endif
		exit(1);
	}
	
	fin_name  = (char *) argv[1];
	fout_name = (char *) argv[2];

	count_in  = strtol( argv[3], NULL, 10 );
	count_out = strtol( argv[4], NULL, 10 );

	fdi = open(fin_name, O_RDONLY);
	fdo = open(fout_name, O_WRONLY | O_CREAT, 0666);

	if ( fdi == -1 ) {
#ifdef DEBUG_ON
		printf("[ERR] Cannot open file %s. Exiting with status %d\n", fin_name, STATUS_ERROR);
#else
		printf("%d\n", STATUS_ERROR);
#endif
		exit(1);
	}
	
	if ( fdo == -1 ) {
#ifdef DEBUG_ON
		printf("[ERR] Cannot open file %s. Exiting with status %d\n", fout_name, STATUS_ERROR);
#else
		printf("%d\n", STATUS_ERROR);
#endif
		exit(1);
	}

	close(fdi);
	close(fdo);
/*----------------------------------------------------------------------------------------------*/

/*---------- Step 2: Set Up the Programmable Logic and the SoC System -------------------------*/

#ifdef DEBUG_ON
	printf("[INFO] Setting up the system\n");
#endif

	// Phase 1 : Prepare the system and fpga fabric

	// system("sudo fpgautil -R"); // Remove the device tree overlay if it is installed
	// program_fpga((char *)C_FIRMWARE_BIN, (char*)C_FIRMWARE_DTBO); // Program the fpga fabric

	system("sudo rmmod -w dma-proxy"); // Remove the dma-proxy module if it is already used
	system("sudo insmod /lib/modules/6.1.30-xilinx-v2023.2/extra/dma-proxy.ko"); // Insert the dma-proxy module into the system

	// Phase 2 : Prepeare the DMA IP Core

	Channel tx_channel, rx_channel;

	tx_channel.name = (char *) TX_CHANNEL_NAME;
	rx_channel.name = (char *) RX_CHANNEL_NAME;

	// Initialize the DMA receiver and transmitter channels
	if (AXI_DMA_Init(&tx_channel) || AXI_DMA_Init(&rx_channel)) {
#ifdef DEBUG_ON
		printf("[ERR] Cannot initialize the DMA IP Core. Exiting with status %d\n", STATUS_ERROR);
#else
		printf("%d\n", STATUS_ERROR);
#endif
		exit(1);
	}

	// Phase 3 : Prepeare the Custom IP Core

	AXILite_Register reg;
	reg.size = C_REG_SIZE;
	reg.base = C_REG_BASE;

	fd_reg = open("/dev/mem", O_RDWR | O_SYNC);
	reg.ptr = mmap(NULL, reg.size, PROT_READ | PROT_WRITE, MAP_SHARED, fd_reg, reg.base);
	close(fd_reg);

	AXILite_Register_Write(reg.ptr, 0x08, 0); // initialize the core

	AXILite_Register_Write(reg.ptr, 0x00, count_in);  // PS-PL transfer words count
	AXILite_Register_Write(reg.ptr, 0x04, count_out); // PL-PS transfer words count

	AXILite_Register_Write(reg.ptr, 0x08, 0xffffffff); // enable the core

	// Check if the registers are truly written
	if ( (AXILite_Register_Read(reg.ptr, 0x00) != count_in) || (AXILite_Register_Read(reg.ptr, 0x04) != count_out) ) {
#ifdef DEBUG_ON
		printf("[ERR] Writting to registers failed. Exiting with status %d\n", STATUS_ERROR);
#else
		printf("%d\n", STATUS_ERROR);
#endif
		close(tx_channel.fd); 
		close(rx_channel.fd);

		exit(1);
	}

	// Phase 4 : Declare the the system is up and running properly

#ifdef DEBUG_ON
	printf("[INFO] Configuration completed successfully\n");
#else
	printf("%d\n", STATUS_RUNNING);
#endif

/*------------------------------------------------------------------------------*/

/*---------- Step 3 : Start the DMA Transactions -------------------------------*/

#ifdef DEBUG_ON
	printf("[INFO] Initiating DMA transaction \n");
#endif

	// Configure the TX thread
	dma_thread_args *tx_args;
	tx_args = (dma_thread_args *)malloc(sizeof(dma_thread_args));
	tx_args->channel 		= &tx_channel;
	tx_args->file 			= fin_name;
	tx_args->transfer_size 	= C_TRANSFER_SIZE;	// in bytes
	tx_args->total_size 	= count_in << 2;		// in bytes
	tx_args->halt_op		= &halt_op;

	// Configure the RX thread
	dma_thread_args *rx_args;
	rx_args = (dma_thread_args *)malloc(sizeof(dma_thread_args));
	rx_args->channel 		= &rx_channel;
	rx_args->file 			= fout_name;
	rx_args->transfer_size 	= C_TRANSFER_SIZE;	// in bytes
	rx_args->total_size 	= count_out << 2;	// in bytes
	rx_args->halt_op		= &halt_op;

	// Create the threads
	pthread_create(&tx_channel.tid, NULL, ps2pl, (void *)tx_args);
	pthread_create(&rx_channel.tid, NULL, pl2ps, (void *)rx_args);	
	
	// Join threads on proper termination or error
	pthread_join(tx_channel.tid, NULL);
	pthread_join(rx_channel.tid, NULL);
        
/*------------------------------------------------------------------------*/

/*------------- Step 4 : Clean Up and Exit -------------------------------*/

	if (( rx_args->status == ERR || tx_args->status == ERR )) {

#ifdef DEBUG_ON
		printf("[ERR] Threads terminated with errors. Exiting with status %d\n", STATUS_ERROR);
#else
		printf("%d\n", STATUS_ERROR);
#endif

	} else {

#ifdef DEBUG_ON
		printf("[INFO] Threads terminated successfully. Exiting with status %d\n", STATUS_COMPLETE);
#else
		printf("%d\n", STATUS_COMPLETE);
#endif

	}

	AXILite_Register_Write(reg.ptr, 0x08, 0); // initialize the core to empty fifos

	// system("sudo fpgautil -R"); // Remove the device tree overlay
	system("sudo rmmod -w dma-proxy"); // Remove the dma-proxy module

	close(tx_channel.fd);
	close(rx_channel.fd);

	free(rx_args);
	free(tx_args);

	return 0;
}
